import numpy as np
import os
from itertools import product
import json
import torch

"""
This script generates synthetic data for a scheduling problem, 
following the methodology described in the paper: 
"Designing dispatching rules with genetic programming for the unrelated machines environment with constraints".

Authors: Kristijan Jaklinovic et al. 
Journal: Expert Systems With Applications
Volume: 172
Year: 2021
"""

def generate_data_batch_rand(batch, pomo,  m, n, process_time_params):
    """
    Generate synthetic data for a scheduling problem.
    
    Parameters:
    m (int): Number of machines.
    n (int): Number of jobs.
    s_max (int): Maximum range of setup times.
    T (float): Due date tightness parameter.
    R (float): Due date range parameter.
    m_p (float): Percentage of jobs eligible on each machine.

    Returns:
    dict: A dictionary containing:
        - p (list): An (m x n) list of processing times.
        - w (list): A list of length n containing the weights of the jobs.
        - setup_times (list): A (m x n x n) list of setup times.
        - due_dates (list): A list of length n containing the due dates of the jobs.
        - release_times (list): A list of length n containing the release times of the jobs.
        - machine_eligibility (list): A list of machines eligible for each job.
    """
    s_max = process_time_params['s_max']
    m_p = process_time_params['m_p']
    
    Processing_list =  []
    Weight_list =  []
    Setup_times_list =  []
    Due_dates_list = []
    Ready_times_list = []
    Eligibility_list = []
    Initial_setup = []
    
    for num in range(batch):
        T = 0.2 + 0.8 * torch.rand(1).item() 
        R = 0.2 + 0.8 * torch.rand(1).item() 

        # Generate random processing times for each machine-job pair.
        p = np.random.randint(0, 100, (m, n))

        # Generate random weights for each job.
        w = np.round(np.random.uniform(0, 1, n), 4)
        
        # Calculate the average processing time across all machine-job pairs.
        p_hat = np.sum(p)/(m*m)
        
        # Generate random release times for each job.
        ready_times = np.random.randint(0, p_hat/2 + 1, n)
        
        # Generate random due dates for each job based on parameters T and R.
        due_dates = []
        for j in range(n): 
            due_tmp = np.random.randint(ready_times[j] + max(0, (p_hat-ready_times[j])*(1-T-R/2)),
                                        ready_times[j] + max(0, (p_hat-ready_times[j])*(1-T+R/2)))
            due_dates.append(due_tmp)
        
        # Generate random setup times for each machine-job-job triplet.
        # Diagonal elements (setup time from a job to itself) are set to 0.
        setup_times = np.random.randint(0, s_max+1, (m, n, n))
        for i in range(m):
            np.fill_diagonal(setup_times[i], 0)
        initial_setup = np.random.randint(0, s_max+1, (m, n)).tolist()

        # Convert numpy arrays to lists for JSON serialization.
        p_list = p.tolist()
        w_list = w.tolist()
        setup_times_list = setup_times.tolist()
        due_dates_list = due_dates
        ready_times_list = ready_times.tolist()

        # Generate machine eligibility matrix and convert to list.
        # Each job is assigned to at least one machine.
        eligibility_matrix = np.zeros((m, n), dtype=bool)
        for machine in range(m):
            percentage = m_p
            num_eligible_jobs = max(int(np.ceil(percentage * n)), 1)
            eligible_jobs = np.random.choice(n, num_eligible_jobs, replace=False)
            eligibility_matrix[machine, eligible_jobs] = True
        
        eligibility_list = []
        for job in range(n):
            eligible_machines = np.where(eligibility_matrix[:, job])[0].tolist()
            if not eligible_machines:
                machine = np.random.choice(m)
                eligibility_matrix[machine, job] = True
                eligible_machines = [machine]
            eligibility_list.append(eligible_machines)

        for _ in range(pomo):
            Processing_list.append(p_list)
            Weight_list.append(w_list)
            Setup_times_list.append(setup_times_list)
            Due_dates_list.append(due_dates_list)
            Ready_times_list.append(ready_times_list)
            Eligibility_list.append(eligibility_list)
            Initial_setup.append(initial_setup)

    # Compile the data into a dictionary.
    data = {
        'p': Processing_list,
        'w': Weight_list,
        'setup_times': Setup_times_list,
        'due_dates': Due_dates_list,
        'release_times': Ready_times_list, 
        'machine_eligibility': Eligibility_list,
        'initial_setup': Initial_setup
    }
    return data    

def generate_data_batch(batch, pomo, m, n, process_time_params):
    """
    Generate synthetic data for a scheduling problem.
    
    Parameters:
    m (int): Number of machines.
    n (int): Number of jobs.
    s_max (int): Maximum range of setup times.
    T (float): Due date tightness parameter.
    R (float): Due date range parameter.
    m_p (float): Percentage of jobs eligible on each machine.

    Returns:
    dict: A dictionary containing:
        - p (list): An (m x n) list of processing times.
        - w (list): A list of length n containing the weights of the jobs.
        - setup_times (list): A (m x n x n) list of setup times.
        - due_dates (list): A list of length n containing the due dates of the jobs.
        - release_times (list): A list of length n containing the release times of the jobs.
        - machine_eligibility (list): A list of machines eligible for each job.
    """
    s_max = process_time_params['s_max'] 
    m_p = process_time_params['m_p'] 
    T = process_time_params['T'] 
    R = process_time_params['R'] 
    
    Processing_list =  []
    Weight_list =  []
    Setup_times_list =  []
    Due_dates_list = []
    Ready_times_list = []
    Eligibility_list = []
    Initial_setup = []
    
    for num in range(batch):
        # Generate random processing times for each machine-job pair.
        p = np.random.randint(0, 100, (m, n))
        # Generate random weights for each job.
        w = np.round(np.random.uniform(0, 1, n), 4)
        # Calculate the average processing time across all machine-job pairs.
        p_hat = np.sum(p)/(m*m)
        
        # Generate random release times for each job.
        ready_times = np.random.randint(0, p_hat/2 + 1, n)
        
        # Generate random due dates for each job based on parameters T and R.
        due_dates = []
        for j in range(n): 
            due_tmp = np.random.randint(ready_times[j] + max(0, (p_hat-ready_times[j])*(1-T-R/2)),
                                        ready_times[j] + max(0, (p_hat-ready_times[j])*(1-T+R/2)))
            due_dates.append(due_tmp)
        
        # Generate random setup times for each machine-job-job triplet.
        # Diagonal elements (setup time from a job to itself) are set to 0.
        setup_times = np.random.randint(0, s_max+1, (m, n, n))
        for i in range(m):
            np.fill_diagonal(setup_times[i], 0)
        initial_setup = np.random.randint(0, s_max+1, (m, n)).tolist()

        # Convert numpy arrays to lists for JSON serialization.
        p_list = p.tolist()
        w_list = w.tolist()
        setup_times_list = setup_times.tolist()
        due_dates_list = due_dates
        ready_times_list = ready_times.tolist()

        # Generate machine eligibility matrix and convert to list.
        # Each job is assigned to at least one machine.
        eligibility_matrix = np.zeros((m, n), dtype=bool)
        for machine in range(m):
            percentage = m_p
            num_eligible_jobs = max(int(np.ceil(percentage * n)), 1)
            eligible_jobs = np.random.choice(n, num_eligible_jobs, replace=False)
            eligibility_matrix[machine, eligible_jobs] = True
        
        eligibility_list = []
        for job in range(n):
            eligible_machines = np.where(eligibility_matrix[:, job])[0].tolist()
            if not eligible_machines:
                machine = np.random.choice(m)
                eligibility_matrix[machine, job] = True
                eligible_machines = [machine]
            eligibility_list.append(eligible_machines)

        for _ in range(pomo):
            Processing_list.append(p_list)
            Weight_list.append(w_list)
            Setup_times_list.append(setup_times_list)
            Due_dates_list.append(due_dates_list)
            Ready_times_list.append(ready_times_list)
            Eligibility_list.append(eligibility_list)
            Initial_setup.append(initial_setup)

    # Compile the data into a dictionary.
    data = {
        'p': Processing_list,
        'w': Weight_list,
        'setup_times': Setup_times_list,
        'due_dates': Due_dates_list,
        'release_times': Ready_times_list, 
        'machine_eligibility': Eligibility_list,
        'initial_setup': Initial_setup
    }
    return data